<?php
// counter.php — لوحة تحكم العيادة مع حقل "التاريخ المرضي" المتراكم

$host    = 'localhost';
$db      = 'queue_system';
$user    = 'root';
$pass    = '';
$charset = 'utf8mb4';

$dsn = "mysql:host=$host;dbname=$db;charset=$charset";
$options = [
    PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
];

try {
    $pdo = new PDO($dsn, $user, $pass, $options);
} catch (Exception $e) {
    die('خطأ في الاتصال بقاعدة البيانات');
}

// محاولة إضافة عمود التاريخ المرضي لو مش موجود
try {
    $pdo->exec("ALTER TABLE patients ADD COLUMN medical_history LONGTEXT NULL");
} catch (Exception $e) {
    // إذا موجود أصلاً نتجاهل الخطأ
}

// أسماء العيادات
$serviceNameMap = [
    1 => "الباطنية",
    2 => "الأطفال",
    3 => "الأنف والأذن",
    4 => "العيون",
    5 => "الجراحة",
    6 => "العظام",
    7 => "الجلدية",
    8 => "النساء",
];

// رقم العيادة من الرابط
$serviceId = isset($_GET['service_id']) ? (int) $_GET['service_id'] : 0;
if (!$serviceId) {
    die('لم يتم تحديد رقم العيادة (مثال: counter.php?service_id=1)');
}
$clinicName = $serviceNameMap[$serviceId] ?? ('عيادة رقم ' . $serviceId);

// دالة تجيب التذكرة الحالية (المُنادى عليها)
function getCurrentTicket(PDO $pdo, int $serviceId)
{
    $sql = "
        SELECT *
        FROM tickets
        WHERE service_id = ?
          AND status = 'called'
        ORDER BY called_at DESC, id DESC
        LIMIT 1
    ";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$serviceId]);
    return $stmt->fetch();
}

// دالة تجيب بيانات المريض حسب رقم الملف
function getPatientByFile(PDO $pdo, string $fileNumber)
{
    if ($fileNumber === '') {
        return null;
    }
    $sql = "SELECT * FROM patients WHERE file_number = ? LIMIT 1";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$fileNumber]);
    return $stmt->fetch() ?: null;
}

$currentTicket = null;
$patient       = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $serviceId   = (int)($_POST['service_id'] ?? 0);
    $currentId   = !empty($_POST['current_id']) ? (int)$_POST['current_id'] : 0;

    $fileNumber  = isset($_POST['file_number'])   ? trim($_POST['file_number'])   : '';
    $diag        = isset($_POST['diagnosis'])     ? trim($_POST['diagnosis'])     : '';
    $meds        = isset($_POST['medications'])   ? trim($_POST['medications'])   : '';
    $docNotes    = isset($_POST['doctor_notes'])  ? trim($_POST['doctor_notes'])  : '';

    // حفظ رقم الملف في التذكرة إن وُجد
    if ($currentId && $fileNumber !== '') {
        $sql = "UPDATE tickets SET file_number = ? WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$fileNumber, $currentId]);
    }

    // لو الدكتور كتب أي شيء، نضيفه إلى التاريخ المرضي
    if ($fileNumber !== '' && ($diag !== '' || $meds !== '' || $docNotes !== '')) {

        // جلب التاريخ المرضي القديم
        $stmt = $pdo->prepare("SELECT medical_history FROM patients WHERE file_number = ? LIMIT 1");
        $stmt->execute([$fileNumber]);
        $row = $stmt->fetch();
        $oldHistory = $row['medical_history'] ?? '';

        // بناء نص الزيارة الجديدة
        $dateNow = date('Y-m-d H:i');  // التاريخ + الوقت
        $parts = [];
        $parts[] = "التاريخ: " . $dateNow;
        if ($diag !== '') {
            $parts[] = "التشخيص: " . $diag;
        }
        if ($meds !== '') {
            $parts[] = "الأدوية: " . $meds;
        }
        if ($docNotes !== '') {
            $parts[] = "ملاحظات الطبيب: " . $docNotes;
        }

        $entryText = implode("\n", $parts) . "\n";
        $divider   = "------------------------------\n";

        // نخلي آخر زيارة في الأعلى
        if ($oldHistory !== '') {
            $newHistory = $entryText . $divider . $oldHistory;
        } else {
            $newHistory = $entryText;
        }

        // تحديث التاريخ المرضي فقط
        $sql = "UPDATE patients SET medical_history = ? WHERE file_number = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$newHistory, $fileNumber]);
    }

    // أزرار التحكم بالتذكرة
    if ($currentId && isset($_POST['finish'])) {
        $sql = "UPDATE tickets SET status = 'finished' WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$currentId]);
    }

    if ($currentId && isset($_POST['skip'])) {
        $sql = "UPDATE tickets SET status = 'skipped' WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$currentId]);
    }

    if ($currentId && isset($_POST['start_visit'])) {
        $sql = "UPDATE tickets SET served_at = NOW() WHERE id = ? AND served_at IS NULL";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$currentId]);
    }

    if ($currentId && isset($_POST['recall'])) {
        $sql = "UPDATE tickets SET called_at = NOW() WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$currentId]);
    }

    if (isset($_POST['next'])) {
        $cur = getCurrentTicket($pdo, $serviceId);
        if (!$cur) {
            $sql = "
                SELECT *
                FROM tickets
                WHERE service_id = ?
                  AND status = 'waiting'
                ORDER BY created_at ASC, id ASC
                LIMIT 1
            ";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([$serviceId]);
            $row = $stmt->fetch();

            if ($row) {
                $sql = "UPDATE tickets SET status = 'called', called_at = NOW(), served_at = NULL WHERE id = ?";
                $up  = $pdo->prepare($sql);
                $up->execute([$row['id']]);
            }
        }
    }

    $currentTicket = getCurrentTicket($pdo, $serviceId);
} else {
    $currentTicket = getCurrentTicket($pdo, $serviceId);
}

// تجهيز متغيرات العرض
$currentCode   = $currentTicket ? htmlspecialchars($currentTicket['full_code']) : 'لا يوجد رقم حالي';
$currentId     = $currentTicket['id'] ?? '';
$currentFile   = $currentTicket['file_number'] ?? '';
$currentServed = $currentTicket['served_at'] ?? null;
$hasCurrent    = !empty($currentId);

// الحقول مقفولة إذا في مريض حالي لكن المعاينة لم تبدأ بعد (served_at فاضي)
$fieldsDisabled = ($hasCurrent && empty($currentServed)) ? 'disabled' : '';

// جلب بيانات المريض لو فيه رقم ملف
$patient       = null;
$patName       = '';
$patMobile     = '';
$patNationalId = '';
$patHistory    = '';

if ($hasCurrent && $currentFile !== '' && $currentFile !== null) {
    $patient = getPatientByFile($pdo, $currentFile);
    if ($patient) {
        $patName       = $patient['full_name']       ?? '';
        $patMobile     = $patient['mobile']          ?? '';
        $patNationalId = $patient['national_id']     ?? '';
        $patHistory    = $patient['medical_history'] ?? '';
    }
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <title>لوحة تحكم العيادة - <?= htmlspecialchars($clinicName) ?></title>
    <style>
        body{
            font-family: Tahoma, Arial, sans-serif;
            background:#f4f4f4;
            text-align:center;
            margin:0;
            padding:0;
            color:#0f172a;
        }
        .container{
            background:#fff;
            width:95%;
            max-width:1200px;
            margin:30px auto;
            padding:25px 20px 30px;
            border-radius:12px;
            box-shadow:0 0 15px rgba(0,0,0,0.08);
            text-align:center;
        }
        h1{
            margin-top:0;
            margin-bottom:5px;
        }
        .clinic-name{
            color:#555;
            margin-bottom:15px;
            font-size:18px;
        }
        .top-row{
            display:flex;
            flex-wrap:wrap;
            gap:20px;
            justify-content:center;
            align-items:flex-start;
        }
        .ticket-box{
            background:#e7f3ff;
            border-radius:10px;
            padding:25px 20px;
            font-size:40px;
            min-width:220px;
            margin-bottom:10px;
        }
        .status-text{
            font-size:15px;
            margin-bottom:10px;
            color:#555;
        }
        .file-row{
            font-size:16px;
            margin-bottom:10px;
        }
        .file-row input{
            font-size:16px;
            padding:5px 8px;
            border-radius:6px;
            border:1px solid #ccc;
            text-align:center;
        }

        .patient-panel{
            flex:2;
            min-width:320px;
            background:#f9fafb;
            border-radius:10px;
            padding:10px 12px 12px;
            border:1px solid #e5e7eb;
            text-align:right;
            direction:rtl;
        }
        .patient-panel h3{
            margin:0 0 6px;
            font-size:16px;
            text-align:center;
        }
        .patient-row{
            margin:2px 0;
            font-size:14px;
        }
        .patient-label{
            font-weight:bold;
            min-width:90px;
            display:inline-block;
        }

        .doctor-block{
            margin-top:12px;
            text-align:right;
            direction:rtl;
        }
        .doctor-block h3{
            margin:0 0 8px;
            font-size:16px;
            text-align:center;
        }
        .doctor-field{
            margin-bottom:8px;
        }
        .doctor-field label{
            display:block;
            font-size:14px;
            margin-bottom:3px;
        }
        .doctor-field textarea{
            width:100%;
            box-sizing:border-box;
            min-height:50px;
            resize:vertical;
            font-size:14px;
            padding:6px 8px;
            border-radius:8px;
            border:1px solid #cbd5e1;
        }

        .history-panel{
            flex:1;
            min-width:260px;
            max-height:380px;
            background:#ffffff;
            border-radius:10px;
            padding:10px 12px;
            border:1px solid #e5e7eb;
            text-align:right;
            direction:rtl;
            overflow-y:auto;
        }
        .history-panel h3{
            margin:0 0 6px;
            font-size:15px;
            text-align:center;
        }
        .history-content{
            font-size:13px;
            white-space:pre-wrap;
        }

        .buttons{
            margin-top:18px;
        }
        .btn{
            display:inline-block;
            min-width:120px;
            padding:9px 16px;
            margin:4px 5px;
            border:none;
            border-radius:999px;
            font-size:16px;
            cursor:pointer;
            color:#fff;
        }
        .next{      background:#27ae60; }
        .recall{    background:#2980b9; }
        .skip{      background:#e67e22; }
        .finish{    background:#e74c3c; }
        .start-vis{ background:#6366f1; }

        .btn:disabled{
            opacity:0.5;
            cursor:not-allowed;
        }

        @media (max-width:900px){
            .top-row{
                flex-direction:column;
                align-items:stretch;
            }
            .history-panel{
                max-height:none;
            }
        }
    </style>
</head>
<body>

<div class="container">
    <h1>لوحة تحكم العيادة</h1>
    <div class="clinic-name"><?= htmlspecialchars($clinicName) ?></div>

    <form method="post">
        <input type="hidden" name="service_id" value="<?= htmlspecialchars($serviceId) ?>">
        <input type="hidden" name="current_id"  value="<?= htmlspecialchars($currentId) ?>">

        <div class="top-row">

            <!-- رقم الدور وحالة المريض -->
            <div>
                <div class="ticket-box">
                    <?= $currentCode ?>
                </div>

                <div class="status-text">
                    <?php if ($currentTicket): ?>
                        الحالة الحالية:
                        <?php if (empty($currentServed)): ?>
                            <strong>تم النداء (بانتظار دخول المريض)</strong>
                        <?php else: ?>
                            <strong>جاري المعاينة داخل العيادة</strong>
                        <?php endif; ?>
                    <?php else: ?>
                        لا يوجد مريض حالي
                    <?php endif; ?>
                </div>

                <div class="file-row">
                    <?php if ($currentTicket): ?>
                        رقم الملف:
                        <?php if ($currentFile !== '' && $currentFile !== null): ?>
                            <?= htmlspecialchars($currentFile) ?>
                            <input type="hidden" name="file_number" value="<?= htmlspecialchars($currentFile) ?>">
                        <?php else: ?>
                            <input type="text" name="file_number" placeholder="اكتب رقم الملف هنا">
                        <?php endif; ?>
                    <?php else: ?>
                        لا يوجد مريض حالي
                    <?php endif; ?>
                </div>
            </div>

            <!-- بيانات المريض + حقول هذه الزيارة -->
            <div class="patient-panel">
                <h3>بيانات المريض</h3>

                <?php if ($currentTicket && $currentFile !== '' && $currentFile !== null): ?>
                    <?php if ($patient): ?>
                        <div class="patient-row">
                            <span class="patient-label">الاسم:</span>
                            <span><?= htmlspecialchars($patName) ?></span>
                        </div>
                        <div class="patient-row">
                            <span class="patient-label">رقم الملف:</span>
                            <span><?= htmlspecialchars($currentFile) ?></span>
                        </div>
                        <?php if ($patNationalId !== ''): ?>
                            <div class="patient-row">
                                <span class="patient-label">الهوية:</span>
                                <span><?= htmlspecialchars($patNationalId) ?></span>
                            </div>
                        <?php endif; ?>
                        <?php if ($patMobile !== ''): ?>
                            <div class="patient-row">
                                <span class="patient-label">الجوال:</span>
                                <span><?= htmlspecialchars($patMobile) ?></span>
                            </div>
                        <?php endif; ?>
                    <?php else: ?>
                        <div class="patient-row">
                            لا توجد بيانات مريض لهذا رقم الملف في قاعدة المرضى.
                        </div>
                    <?php endif; ?>

                    <div class="doctor-block">
                        <h3>بيانات هذه الزيارة</h3>

                        <div class="doctor-field">
                            <label>تشخيص المرض:</label>
                            <textarea name="diagnosis" placeholder="ابدأ المعاينة أولاً لكتابة التشخيص" <?= $fieldsDisabled ?>></textarea>
                        </div>

                        <div class="doctor-field">
                            <label>الأدوية:</label>
                            <textarea name="medications" placeholder="ابدأ المعاينة أولاً لكتابة الأدوية" <?= $fieldsDisabled ?>></textarea>
                        </div>

                        <div class="doctor-field">
                            <label>ملاحظات إضافية:</label>
                            <textarea name="doctor_notes" placeholder="ابدأ المعاينة أولاً لكتابة الملاحظات" <?= $fieldsDisabled ?>></textarea>
                        </div>
                    </div>

                <?php elseif ($currentTicket): ?>
                    <div class="patient-row">
                        أدخل رقم الملف أولاً ليتم جلب بيانات المريض.
                    </div>
                <?php else: ?>
                    <div class="patient-row">
                        لا يوجد مريض حالي.
                    </div>
                <?php endif; ?>
            </div>

            <!-- لوحة التاريخ المرضي -->
            <div class="history-panel">
                <h3>التاريخ المرضي</h3>
                <div class="history-content">
                    <?php if ($patHistory !== ''): ?>
                        <?= nl2br(htmlspecialchars($patHistory)) ?>
                    <?php else: ?>
                        لا يوجد تاريخ مرضي مسجّل بعد.
                    <?php endif; ?>
                </div>
            </div>

        </div>

        <div class="buttons">
            <button class="btn next" type="submit" name="next"
                <?= $hasCurrent ? 'disabled' : '' ?>>
                التالي
            </button>

            <button class="btn recall" type="submit" name="recall"
                <?= $hasCurrent ? '' : 'disabled' ?>>
                إعادة النداء
            </button>

            <button class="btn start-vis" type="submit" name="start_visit"
                <?= ($hasCurrent && empty($currentServed)) ? '' : 'disabled' ?>>
                بدء المعاينة
            </button>

            <button class="btn skip" type="submit" name="skip"
                <?= $hasCurrent ? '' : 'disabled' ?>>
                تخطي
            </button>

            <button class="btn finish" type="submit" name="finish"
                <?= $hasCurrent ? '' : 'disabled' ?>>
                إنهاء المعاينة
            </button>
        </div>
    </form>
</div>

</body>
</html>
