<?php
session_start();

if (empty($_SESSION['logged_in'])) {
    header("Location: login.php");
    exit;
}
?>

<?php
// dashboard.php — لوحة متابعة نظام الدور (يومي + AJAX + ازدحام + تأخير)

// إعداد الاتصال بقاعدة البيانات
$host    = 'localhost';
$db      = 'queue_system';
$user    = 'root';
$pass    = '';
$charset = 'utf8mb4';

$dsn = "mysql:host=$host;dbname=$db;charset=$charset";
$options = [
    PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
];

try {
    $pdo = new PDO($dsn, $user, $pass, $options);
} catch (Exception $e) {
    die('خطأ في الاتصال بقاعدة البيانات');
}

// أسماء العيادات
$serviceNameMap = [
    1 => "الباطنية",
    2 => "الأطفال",
    3 => "الأنف والأذن",
    4 => "العيون",
    5 => "الجراحة",
    6 => "العظام",
    7 => "الجلدية",
    8 => "النساء",
];

// دالة تجيب إحصائيات (لليوم فقط) لكل عيادة
function getStatsByService(PDO $pdo): array
{
    $sql = "
        SELECT
            service_id,
            SUM(CASE WHEN status = 'waiting'  THEN 1 ELSE 0 END) AS waiting_count,
            SUM(CASE WHEN status = 'called'   THEN 1 ELSE 0 END) AS called_count,
            SUM(CASE WHEN status = 'finished' THEN 1 ELSE 0 END) AS finished_count,
            SUM(CASE WHEN status = 'skipped'  THEN 1 ELSE 0 END) AS skipped_count,
            COUNT(*) AS total_count,
            MAX(called_at) AS last_called_at,
            MAX(
                CASE WHEN status = 'waiting'
                     THEN TIMESTAMPDIFF(MINUTE, created_at, NOW())
                     ELSE NULL
                END
            ) AS max_wait_min
        FROM tickets
        WHERE DATE(created_at) = CURDATE()
        GROUP BY service_id
    ";
    $stmt = $pdo->query($sql);
    $rows = $stmt->fetchAll();

    $result = [];
    foreach ($rows as $r) {
        $sid = (int)$r['service_id'];
        $result[$sid] = [
            'waiting'      => (int)($r['waiting_count']  ?? 0),
            'called'       => (int)($r['called_count']   ?? 0),
            'finished'     => (int)($r['finished_count'] ?? 0),
            'skipped'      => (int)($r['skipped_count']  ?? 0),
            'total'        => (int)($r['total_count']    ?? 0),
            'last_call'    => $r['last_called_at'] ?? null,
            'max_wait_min' => isset($r['max_wait_min']) ? (int)$r['max_wait_min'] : 0,
        ];
    }
    return $result;
}

// معالجة طلبات AJAX (ترجع JSON فقط)
if (isset($_GET['ajax']) && $_GET['ajax'] == '1') {
    header('Content-Type: application/json; charset=utf-8');
    $stats = getStatsByService($pdo);
    echo json_encode($stats);
    exit;
}

$resetMsg = "";

// زر تصفير كل الأرقام (يحذف تذاكر اليوم وكل الأيام)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['reset_all'])) {
    $pdo->exec("TRUNCATE TABLE tickets");
    $resetMsg = "✅ تم تصفير أرقام التذاكر لجميع العيادات.";
}

// إحصائيات أولية لعرضها عند تحميل الصفحة
$statsByService = getStatsByService($pdo);
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
<meta charset="UTF-8">
<title>لوحة متابعة نظام الدور</title>

<style>
    body{
        font-family: Tahoma, Arial, sans-serif;
        background:#f3f4f6;
        margin:0;
        padding:0;
        color:#111827;
    }
    .wrapper{
        max-width:1200px;
        margin:25px auto 40px;
        padding:0 15px;
    }
    h1{
        text-align:center;
        margin:0 0 6px;
        font-size:26px;
    }
    .subtitle{
        text-align:center;
        font-size:14px;
        color:#6b7280;
        margin-bottom:15px;
    }

    .top-bar{
        display:flex;
        justify-content:space-between;
        align-items:center;
        gap:10px;
        margin-bottom:15px;
        flex-wrap:wrap;
    }
    .reset-form{margin:0;}
    .reset-btn{
        padding:7px 16px;
        border:none;
        border-radius:999px;
        background:#ef4444;
        color:#fff;
        cursor:pointer;
        font-size:13px;
    }
    .reset-btn:active{transform:scale(.97);}
    .info{
        font-size:13px;
        color:#6b7280;
    }
    .msg{
        margin:10px 0;
        padding:8px 10px;
        border-radius:8px;
        font-size:13px;
        background:#ecfdf3;
        color:#166534;
        border:1px solid #bbf7d0;
        text-align:right;
    }

    /* ملخص اليوم */
    .summary{
        margin-bottom:15px;
        background:#ffffff;
        border-radius:12px;
        padding:10px 12px;
        border:1px solid #e5e7eb;
        box-shadow:0 4px 10px rgba(0,0,0,0.03);
    }
    .summary-title{
        font-size:14px;
        font-weight:bold;
        margin-bottom:6px;
    }
    .summary-grid{
        display:grid;
        grid-template-columns:repeat(auto-fit,minmax(140px,1fr));
        gap:8px;
        font-size:13px;
    }
    .summary-item{
        background:#f9fafb;
        border-radius:8px;
        padding:6px 8px;
    }
    .summary-label{color:#4b5563;font-size:12px;margin-bottom:2px;}
    .summary-value{font-weight:bold;font-size:16px;}

    /* كروت العيادات */
    .grid{
        display:grid;
        grid-template-columns:repeat(auto-fit,minmax(240px,1fr));
        gap:12px;
        margin-top:10px;
    }
    .card{
        background:#ffffff;
        border-radius:12px;
        padding:14px 12px 12px;
        border:1px solid #e5e7eb;
        box-shadow:0 4px 10px rgba(0,0,0,0.03);
    }
    .clinic-name{
        font-size:16px;
        font-weight:bold;
        margin-bottom:4px;
    }
    .clinic-sub{
        font-size:12px;
        color:#6b7280;
        margin-bottom:8px;
    }
    .row{
        display:flex;
        justify-content:space-between;
        font-size:13px;
        margin:2px 0;
    }
    .row span.label{color:#4b5563;}
    .row span.value{font-weight:bold;}
    .pill{
        display:inline-block;
        padding:2px 8px;
        border-radius:999px;
        font-size:11px;
        background:#f3f4ff;
        color:#4f46e5;
        margin-top:4px;
    }
    .pill.delay{
        background:#fef2f2;
        color:#b91c1c;
    }
    .pill.ok{
        background:#ecfdf3;
        color:#166534;
    }
    .small-link{
        font-size:12px;
        margin-top:6px;
        display:block;
    }
    .small-link a{
        color:#2563eb;
        text-decoration:none;
    }

    /* أقسام إضافية: ازدحام/تأخير */
    .extra-sections{
        display:grid;
        grid-template-columns:repeat(auto-fit,minmax(260px,1fr));
        gap:10px;
        margin:10px 0 5px;
    }
    .extra-card{
        background:#ffffff;
        border-radius:12px;
        padding:10px 12px;
        border:1px solid #e5e7eb;
        box-shadow:0 4px 10px rgba(0,0,0,0.03);
        font-size:13px;
    }
    .extra-title{
        font-weight:bold;
        margin-bottom:5px;
    }
    .extra-list{
        list-style:none;
        padding:0;
        margin:0;
    }
    .extra-list li{
        padding:3px 0;
        border-bottom:1px dashed #e5e7eb;
    }
    .extra-list li:last-child{
        border-bottom:none;
    }
</style>
</head>
<body>
<div class="wrapper">
    <h1>لوحة متابعة نظام الدور</h1>
    <div class="subtitle">
        إحصائيات اليوم فقط (من بداية اليوم حتى الآن) — يتم تحديث الأرقام تلقائياً كل <strong>10 ثوانٍ</strong>.
    </div>

    <div class="top-bar">
        <div class="info">
            يمكن تصفير جميع الأرقام في بداية اليوم، ليبدأ الترقيم من جديد لكل العيادات.
        </div>
        <form method="post" class="reset-form"
              onsubmit="return confirm('هل أنت متأكد من تصفير جميع أرقام التذاكر؟ سيتم حذف كل السجلات.');">
            <button type="submit" name="reset_all" class="reset-btn">تصفير جميع الأرقام</button>
        </form>
    </div>

    <?php if ($resetMsg): ?>
        <div class="msg"><?= htmlspecialchars($resetMsg) ?></div>
    <?php endif; ?>

    <!-- ملخص اليوم -->
    <div class="summary">
        <div class="summary-title">إحصائيات اليوم</div>
        <div class="summary-grid">
            <div class="summary-item">
                <div class="summary-label">إجمالي عدد التذاكر</div>
                <div class="summary-value" id="sum-total">0</div>
            </div>
            <div class="summary-item">
                <div class="summary-label">عدد المرضى المنتظرين</div>
                <div class="summary-value" id="sum-waiting">0</div>
            </div>
            <div class="summary-item">
                <div class="summary-label">عدد المرضى المنتهين</div>
                <div class="summary-value" id="sum-finished">0</div>
            </div>
            <div class="summary-item">
                <div class="summary-label">عدد الحالات المتخطيّة</div>
                <div class="summary-value" id="sum-skipped">0</div>
            </div>
            <div class="summary-item">
                <div class="summary-label">عدد العيادات النشطة اليوم</div>
                <div class="summary-value" id="sum-active-clinics">0</div>
            </div>
        </div>
    </div>

    <!-- أقسام إضافية: ازدحام وتأخير -->
    <div class="extra-sections">
        <div class="extra-card">
            <div class="extra-title">أكثر العيادات ازدحاماً اليوم</div>
            <ul class="extra-list" id="top-busy-list">
                <li>جاري التحميل...</li>
            </ul>
        </div>
        <div class="extra-card">
            <div class="extra-title">العيادات التي تأخرت في نداء المرضى</div>
            <small style="color:#6b7280;display:block;margin-bottom:4px;">
                تُعتبر العيادة متأخرة إذا كان أقدم مريض منتظر منذ أكثر من 20 دقيقة.
            </small>
            <ul class="extra-list" id="delayed-list">
                <li>جاري التحميل...</li>
            </ul>
        </div>
    </div>

    <!-- كروت العيادات -->
    <div class="grid" id="clinics-grid">
        <?php foreach ($serviceNameMap as $sid => $name):
            $s = $statsByService[$sid] ?? [
                'waiting' => 0, 'called' => 0,
                'finished' => 0, 'skipped' => 0,
                'total' => 0, 'last_call' => null, 'max_wait_min' => 0
            ];
        ?>
            <div class="card" data-service-id="<?= (int)$sid ?>">
                <div class="clinic-name"><?= htmlspecialchars($name) ?></div>
                <div class="clinic-sub">عيادة رقم <?= (int)$sid ?></div>

                <div class="row">
                    <span class="label">المنتظرون:</span>
                    <span class="value value-waiting"><?= (int)$s['waiting'] ?></span>
                </div>
                <div class="row">
                    <span class="label">قيد النداء (called):</span>
                    <span class="value value-called"><?= (int)$s['called'] ?></span>
                </div>
                <div class="row">
                    <span class="label">مُنتهون:</span>
                    <span class="value value-finished"><?= (int)$s['finished'] ?></span>
                </div>
                <div class="row">
                    <span class="label">متخطيّن:</span>
                    <span class="value value-skipped"><?= (int)$s['skipped'] ?></span>
                </div>
                <div class="row">
                    <span class="label">إجمالي اليوم:</span>
                    <span class="value value-total"><?= (int)$s['total'] ?></span>
                </div>
                <div class="row">
                    <span class="label">أقصى وقت انتظار:</span>
                    <span class="value value-max-wait">
                        <?= $s['max_wait_min'] > 0 ? $s['max_wait_min'] . ' دقيقة' : '—' ?>
                    </span>
                </div>

                <div class="pill last-call-pill">
                    آخر نداء:
                    <span class="value-last-call">
                        <?= $s['last_call'] ? htmlspecialchars($s['last_call']) : '—' ?>
                    </span>
                </div>
                <div class="pill status-pill ok">
                    <span class="value-delay-text">
                        حالة العيادة: طبيعية
                    </span>
                </div>

                <span class="small-link">
                    <a href="counter.php?service_id=<?= (int)$sid ?>" target="_blank">
                        فتح لوحة العيادة
                    </a>
                </span>
            </div>
        <?php endforeach; ?>
    </div>
</div>

<script>
// أسماء العيادات في الجافاسكربت
const serviceNameMap = <?php echo json_encode($serviceNameMap, JSON_UNESCAPED_UNICODE); ?>;

// جلب الإحصائيات من السيرفر
function fetchDashboardStats() {
    return fetch('dashboard.php?ajax=1')
        .then(res => res.json());
}

// تحديث الكروت + الملخص + أقسام الازدحام/التأخير
function updateDashboardUI(data) {
    // 1) تحديث كروت العيادات
    document.querySelectorAll('.card[data-service-id]').forEach(card => {
        const sid = card.getAttribute('data-service-id');
        const s = data[sid] || {
            waiting:0, called:0, finished:0, skipped:0,
            total:0, last_call:null, max_wait_min:0
        };

        card.querySelector('.value-waiting').textContent  = s.waiting;
        card.querySelector('.value-called').textContent   = s.called;
        card.querySelector('.value-finished').textContent = s.finished;
        card.querySelector('.value-skipped').textContent  = s.skipped;
        card.querySelector('.value-total').textContent    = s.total;
        card.querySelector('.value-max-wait').textContent =
            s.max_wait_min > 0 ? (s.max_wait_min + ' دقيقة') : '—';
        card.querySelector('.value-last-call').textContent =
            s.last_call ? s.last_call : '—';

        // حالة التأخير لكل عيادة
        const statusPill = card.querySelector('.status-pill');
        const delayText  = card.querySelector('.value-delay-text');
        const delayed = (s.waiting > 0 && s.max_wait_min >= 20);

        if (delayed) {
            statusPill.classList.remove('ok');
            statusPill.classList.add('delay');
            delayText.textContent = 'حالة العيادة: متأخرة';
        } else {
            statusPill.classList.remove('delay');
            statusPill.classList.add('ok');
            delayText.textContent = 'حالة العيادة: طبيعية';
        }
    });

    // 2) ملخص اليوم
    let sumTotal = 0, sumWaiting = 0, sumFinished = 0, sumSkipped = 0, activeClinics = 0;
    const busyList = [];
    const delayedList = [];

    Object.keys(serviceNameMap).forEach(sid => {
        const s = data[sid] || {
            waiting:0, called:0, finished:0, skipped:0,
            total:0, last_call:null, max_wait_min:0
        };

        sumTotal   += s.total;
        sumWaiting += s.waiting;
        sumFinished+= s.finished;
        sumSkipped += s.skipped;
        if (s.total > 0) activeClinics++;

        if (s.waiting > 0) {
            busyList.push({sid, waiting: s.waiting});
        }
        if (s.waiting > 0 && s.max_wait_min >= 20) {
            delayedList.push({sid, wait: s.max_wait_min});
        }
    });

    document.getElementById('sum-total').textContent          = sumTotal;
    document.getElementById('sum-waiting').textContent        = sumWaiting;
    document.getElementById('sum-finished').textContent       = sumFinished;
    document.getElementById('sum-skipped').textContent        = sumSkipped;
    document.getElementById('sum-active-clinics').textContent = activeClinics;

    // 3) أكثر العيادات ازدحاماً (ترتيب حسب waiting)
    const topBusyEl = document.getElementById('top-busy-list');
    topBusyEl.innerHTML = '';
    busyList.sort((a,b) => b.waiting - a.waiting);
    const top3 = busyList.slice(0,3);
    if (top3.length === 0) {
        topBusyEl.innerHTML = '<li>لا يوجد ازدحام حالياً.</li>';
    } else {
        top3.forEach(item => {
            const li = document.createElement('li');
            li.textContent = serviceNameMap[item.sid] + ' — ' + item.waiting + ' مريض منتظر';
            topBusyEl.appendChild(li);
        });
    }

    // 4) العيادات المتأخرة
    const delayedEl = document.getElementById('delayed-list');
    delayedEl.innerHTML = '';
    if (delayedList.length === 0) {
        delayedEl.innerHTML = '<li>لا توجد عيادات متأخرة حالياً.</li>';
    } else {
        delayedList
            .sort((a,b) => b.wait - a.wait) // الأكثر تأخيراً أولاً
            .forEach(item => {
                const li = document.createElement('li');
                li.textContent =
                    serviceNameMap[item.sid] + ' — أقدم مريض ينتظر منذ ' +
                    item.wait + ' دقيقة تقريباً';
                delayedEl.appendChild(li);
            });
    }
}

// جلب وتحديث
function refreshDashboard() {
    fetchDashboardStats()
        .then(data => updateDashboardUI(data))
        .catch(err => console.error('خطأ في جلب بيانات الداشبورد:', err));
}

// أول تحميل
refreshDashboard();
// كل 10 ثواني
setInterval(refreshDashboard, 10000);
</script>
</body>
</html>
