<?php
// kiosk.php — كشك أخذ الأرقام (مع البحث عن المريض)

$host    = 'localhost';
$db      = 'queue_system';
$user    = 'root';
$pass    = '';
$charset = 'utf8mb4';

$dsn = "mysql:host=$host;dbname=$db;charset=$charset";
$options = [
    PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
];

try {
    $pdo = new PDO($dsn, $user, $pass, $options);
} catch (Exception $e) {
    die('خطأ في الاتصال بقاعدة البيانات');
}

// أسماء العيادات
$serviceNameMap = [
    1 => "الباطنية",
    2 => "الأطفال",
    3 => "الأنف والأذن",
    4 => "العيون",
    5 => "الجراحة",
    6 => "العظام",
    7 => "الجلدية",
    8 => "النساء",
];

// حرف لكل عيادة
$serviceLetterMap = [
    1 => 'A',
    2 => 'B',
    3 => 'C',
    4 => 'D',
    5 => 'E',
    6 => 'F',
    7 => 'G',
    8 => 'H',
];

$step = 'choose_clinic';

$ticketCreated      = false;
$createdServiceId   = null;
$createdServiceName = '';
$createdLetter      = '';
$createdFullCode    = '';
$createdFileNumber  = '';

$selectedServiceId   = null;
$selectedServiceName = '';
$selectedLetter      = '';

$patient      = null;
$searchError  = '';
$searchType   = 'national_id';
$searchValue  = '';

// معالجة الطلب
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    // 1) اختيار العيادة
    if ($action === 'choose_service' && isset($_POST['service_id'])) {
        $serviceId = (int)$_POST['service_id'];

        if ($serviceId <= 0 || !isset($serviceNameMap[$serviceId])) {
            die('رقم العيادة غير صحيح');
        }

        $selectedServiceId   = $serviceId;
        $selectedServiceName = $serviceNameMap[$serviceId];
        $selectedLetter      = $serviceLetterMap[$serviceId] ?? 'X';

        $step = 'search_patient';
    }

    // 2) البحث عن المريض
    elseif ($action === 'search_patient' && isset($_POST['service_id'])) {
        $serviceId = (int)$_POST['service_id'];

        if ($serviceId <= 0 || !isset($serviceNameMap[$serviceId])) {
            die('رقم العيادة غير صحيح');
        }

        $selectedServiceId   = $serviceId;
        $selectedServiceName = $serviceNameMap[$serviceId];
        $selectedLetter      = $serviceLetterMap[$serviceId] ?? 'X';

        $searchType  = $_POST['search_type'] ?? 'national_id';
        $searchValue = trim($_POST['search_value'] ?? '');

        if ($searchValue === '') {
            $searchError = 'الرجاء إدخال قيمة للبحث.';
            $step = 'search_patient';
        } else {
            if ($searchType === 'national_id') {
                $sql = "SELECT * FROM patients WHERE national_id = ? LIMIT 1";
            } elseif ($searchType === 'file_number') {
                $sql = "SELECT * FROM patients WHERE file_number = ? LIMIT 1";
            } elseif ($searchType === 'mobile') {
                $sql = "SELECT * FROM patients WHERE mobile = ? LIMIT 1";
            } else {
                $sql = "SELECT * FROM patients WHERE national_id = ? LIMIT 1";
            }

            $stmt = $pdo->prepare($sql);
            $stmt->execute([$searchValue]);
            $patient = $stmt->fetch();

            if (!$patient) {
                $searchError = '❌ لم يتم العثور على مريض بهذه البيانات.';
            }

            $step = 'search_patient';
        }
    }

    // 3) تأكيد وإصدار التذكرة
    elseif ($action === 'confirm_ticket' && isset($_POST['service_id'], $_POST['patient_id'])) {
        $serviceId = (int)$_POST['service_id'];
        $patientId = (int)$_POST['patient_id'];

        if ($serviceId <= 0 || !isset($serviceNameMap[$serviceId])) {
            die('رقم العيادة غير صحيح');
        }

        $selectedServiceId   = $serviceId;
        $selectedServiceName = $serviceNameMap[$serviceId];
        $selectedLetter      = $serviceLetterMap[$serviceId] ?? 'X';

        // جلب رقم الملف من جدول المرضى
        $stmt = $pdo->prepare("SELECT file_number FROM patients WHERE id = ? LIMIT 1");
        $stmt->execute([$patientId]);
        $p = $stmt->fetch();

        if (!$p) {
            die('المريض غير موجود.');
        }

        $createdFileNumber = $p['file_number'];

        // حساب رقم التسلسل للتذكرة
        $sql  = "SELECT COUNT(*) AS cnt FROM tickets WHERE service_id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$serviceId]);
        $row  = $stmt->fetch();
        $cnt  = (int)($row['cnt'] ?? 0);

        $nextNumber = $cnt + 1;

        // كود التذكرة مثل A001
        $letter   = $selectedLetter;
        $fullCode = sprintf('%s%03d', $letter, $nextNumber);

        // إدخال التذكرة
        $insertSql = "
            INSERT INTO tickets (service_id, full_code, status, created_at, file_number)
            VALUES (?, ?, 'waiting', NOW(), ?)
        ";
        $insertStmt = $pdo->prepare($insertSql);
        $insertStmt->execute([$serviceId, $fullCode, $createdFileNumber]);

        $ticketCreated      = true;
        $createdServiceId   = $serviceId;
        $createdServiceName = $selectedServiceName;
        $createdLetter      = $letter;
        $createdFullCode    = $fullCode;

        $step = 'ticket_issued';
    }
}

// تقسيم العيادات عمودين يمين / يسار مع شعار بالنص
$services = array_keys($serviceNameMap);
$total    = count($services);
$half     = (int)ceil($total / 2);
$leftServices  = array_slice($services, 0, $half);
$rightServices = array_slice($services, $half);
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
<meta charset="UTF-8">
<title>نظام الدور - كشك الأرقام</title>

<style>
body{
    font-family: Tahoma, Arial, sans-serif;
    background:#ffffff;
    margin:0;
    padding:0;
    color:#0f172a;
}
.wrapper{
    max-width:1200px;
    margin:0 auto;
    padding:25px 15px 35px;
    text-align:center;
}
h1{
    margin:0 0 10px;
    font-size:32px;
    color:#0f172a;
}
h2{
    margin:5px 0 20px;
    font-size:18px;
    font-weight:normal;
    color:#334155;
}

/* توزيع العيادات عمودين + شعار بالنص */
.layout{
    display:flex;
    justify-content:center;
    align-items:flex-start;
    gap:35px;
    margin-top:10px;
}
.column{
    flex:1;
    display:grid;
    grid-template-columns:1fr;
    gap:12px;
}
.clinic-card{
    background:#f8fafc;
    border-radius:14px;
    padding:4px;
    border:1px solid #e2e8f0;
}
.clinic-btn{
    width:100%;
    border:none;
    background:linear-gradient(135deg,#bfdbfe,#93c5fd);
    border-radius:10px;
    color:#0f172a;
    padding:14px 10px;
    cursor:pointer;
    font-size:18px;
    font-weight:bold;
    display:flex;
    flex-direction:column;
}
.clinic-btn span.small{
    font-size:12px;
    opacity:.8;
}
.clinic-btn:active{
    transform:scale(.97);
}
.logo-box{
    width:280px;
    display:flex;
    justify-content:center;
    align-items:center;
}
.logo-box img{
    width:100%;
    opacity:1;
    filter:none;
}

/* صندوق البحث عن المريض */
.search-box{
    margin-top:25px;
    max-width:480px;
    margin-inline:auto;
    background:#ffffff;
    border-radius:16px;
    padding:20px 18px 22px;
    border:1px solid #e2e8f0;
    box-shadow:0 8px 22px rgba(0,0,0,.05);
    text-align:right;
}
.search-title{
    text-align:center;
    font-size:18px;
    margin-bottom:10px;
}
.search-options{
    margin-bottom:8px;
    font-size:14px;
}
.search-options label{
    margin-left:10px;
}
.search-input{
    width:100%;
    padding:8px 10px;
    font-size:16px;
    border-radius:10px;
    border:1px solid #cbd5e1;
    text-align:center;
    margin-bottom:10px;
}
.search-submit{
    width:100%;
    padding:9px;
    border:none;
    border-radius:999px;
    background:linear-gradient(135deg,#bfdbfe,#60a5fa);
    color:#0f172a;
    font-size:15px;
    font-weight:bold;
    cursor:pointer;
}
.search-error{
    margin-top:8px;
    color:#b91c1c;
    font-size:13px;
    text-align:center;
}

/* بيانات المريض */
.patient-box{
    margin-top:10px;
    background:#f9fafb;
    border-radius:12px;
    padding:10px 12px;
    border:1px solid #e5e7eb;
    text-align:right;
    font-size:14px;
}
.patient-row{
    margin:2px 0;
}

/* زر تأكيد التذكرة */
.confirm-btn{
    margin-top:10px;
    width:100%;
    padding:9px;
    border:none;
    border-radius:999px;
    background:linear-gradient(135deg,#86efac,#4ade80);
    color:#065f46;
    font-size:15px;
    font-weight:bold;
    cursor:pointer;
}

/* 🎟 شكل التذكرة على الشاشة */
.ticket-box{
    margin:25px auto;
    max-width:380px;
    background:#ffffff;
    border-radius:16px;
    border:1px solid #e2e8f0;
    padding:18px 14px 20px;
    position:relative;
    overflow:hidden;
}
.ticket-box::before{
    content:"";
    position:absolute;
    inset:0;
    background:url("images/logo.png") no-repeat center;
    background-size:75%;
    opacity:.08;
    pointer-events:none;
}
.ticket-header{
    font-size:16px;
    font-weight:bold;
    margin-bottom:8px;
    text-align:center;
}
.ticket-number{
    font-size:64px;
    font-weight:bold;
    margin:6px 0;
    text-align:center;
}
.ticket-line{
    text-align:center;
    font-size:16px;
    margin:3px 0;
}
.ticket-footer{
    margin-top:10px;
    font-size:15px;
    font-weight:bold;
    text-align:center;
}

/* زر الرجوع */
.back-link{
    display:inline-block;
    margin-top:10px;
    padding:9px 18px;
    border-radius:999px;
    background:#e2e8f0;
    color:#0f172a;
    text-decoration:none;
    font-size:13px;
}

/* 🖨️ إعداد الطباعة على مقاس طابعة الكشك */
@media print{

    body *{
        visibility: hidden;
    }

    .ticket-box,
    .ticket-box *{
        visibility: visible;
    }

    .ticket-box{
        position:absolute;
        top:0;
        left:0;
        right:auto;
        margin:0;
        width:58mm;
        min-height:90mm;
        border:none;
        border-radius:0;
        box-shadow:none;
        padding:10px;
    }

    .ticket-box::before{
        opacity:.25;
        background-size:90%;
    }

    .ticket-header{
        font-size:18px;
        margin-bottom:6px;
    }
    .ticket-number{
        font-size:54px;
        margin:4px 0;
    }
    .ticket-line{
        font-size:18px;
        margin:3px 0;
    }
    .ticket-footer{
        font-size:16px;
        margin-top:8px;
    }

    .back-link{
        display:none !important;
    }
}

@media (max-width:900px){
    .layout{
        flex-direction:column;
        align-items:center;
    }
    .logo-box{
        width:200px;
    }
}
</style>

<?php if ($step === 'ticket_issued' && $ticketCreated): ?>
<script>
// طباعة تلقائية عند إصدار التذكرة ثم العودة للكشك
window.onload = function() {
    window.print();
    setTimeout(function(){
        window.location.href = "kiosk.php";
    }, 800);
};
</script>
<?php endif; ?>

</head>
<body>
<div class="wrapper">
    <h1>نظام إدارة الدور</h1>

    <?php if ($step === 'ticket_issued' && $ticketCreated): ?>

        <div class="ticket-box">
            <div class="ticket-header">شركة القمر الذهبي</div>

            <div class="ticket-number">
                <?= htmlspecialchars($createdFullCode) ?>
            </div>

            <div class="ticket-line">
                العيادة: <?= htmlspecialchars($createdServiceName) ?> (<?= htmlspecialchars($createdLetter) ?>)
            </div>
            <div class="ticket-line">
                رقم الملف: <?= htmlspecialchars($createdFileNumber) ?>
            </div>

            <div class="ticket-footer">
                شكراً لاختياركم لنا
            </div>
        </div>

    <?php elseif ($step === 'search_patient' && $selectedServiceId !== null): ?>

        <h2>العيادة: <?= htmlspecialchars($selectedServiceName) ?></h2>

        <div class="search-box">
            <div class="search-title">البحث عن المريض</div>

            <form method="post">
                <input type="hidden" name="action" value="search_patient">
                <input type="hidden" name="service_id" value="<?= (int)$selectedServiceId ?>">

                <div class="search-options">
                    <label>
                        <input type="radio" name="search_type" value="national_id"
                            <?= $searchType === 'national_id' ? 'checked' : '' ?>>
                        رقم الهوية
                    </label>
                    <label>
                        <input type="radio" name="search_type" value="file_number"
                            <?= $searchType === 'file_number' ? 'checked' : '' ?>>
                        رقم الملف
                    </label>
                    <label>
                        <input type="radio" name="search_type" value="mobile"
                            <?= $searchType === 'mobile' ? 'checked' : '' ?>>
                        رقم الجوال
                    </label>
                </div>

                <input
                    type="text"
                    name="search_value"
                    class="search-input"
                    placeholder="أدخل القيمة هنا"
                    value="<?= htmlspecialchars($searchValue) ?>"
                    required
                >

                <button type="submit" class="search-submit">بحث</button>

                <?php if ($searchError): ?>
                    <div class="search-error"><?= htmlspecialchars($searchError) ?></div>
                <?php endif; ?>
            </form>

            <?php if ($patient): ?>
                <div class="patient-box">
                    <div class="patient-row"><strong>الاسم:</strong> <?= htmlspecialchars($patient['full_name']) ?></div>
                    <div class="patient-row"><strong>رقم الملف:</strong> <?= htmlspecialchars($patient['file_number']) ?></div>
                    <div class="patient-row"><strong>الجوال:</strong> <?= htmlspecialchars($patient['mobile']) ?></div>
                    <div class="patient-row"><strong>رقم الهوية:</strong> <?= htmlspecialchars($patient['national_id'] ?? '') ?></div>
                </div>

                <form method="post">
                    <input type="hidden" name="action" value="confirm_ticket">
                    <input type="hidden" name="service_id" value="<?= (int)$selectedServiceId ?>">
                    <input type="hidden" name="patient_id" value="<?= (int)$patient['id'] ?>">
                    <button type="submit" class="confirm-btn">تأكيد وإصدار رقم الدور</button>
                </form>
            <?php endif; ?>

            <div style="text-align:center;margin-top:10px;">
                <a href="kiosk.php" class="back-link">رجوع لاختيار عيادة أخرى</a>
            </div>
        </div>

    <?php else: ?>

        <h2>اختر العيادة</h2>

        <div class="layout">

            <!-- عمود يسار -->
            <div class="column">
                <?php foreach ($leftServices as $id): ?>
                    <form method="post" class="clinic-card">
                        <input type="hidden" name="action" value="choose_service">
                        <input type="hidden" name="service_id" value="<?= (int)$id ?>">
                        <button type="submit" class="clinic-btn">
                            <span><?= htmlspecialchars($serviceNameMap[$id]) ?></span>
                            <span class="small">الحرف: <?= htmlspecialchars($serviceLetterMap[$id]) ?></span>
                        </button>
                    </form>
                <?php endforeach; ?>
            </div>

            <!-- الشعار في النص -->
            <div class="logo-box">
                <img src="images/logo.png" alt="Logo">
            </div>

            <!-- عمود يمين -->
            <div class="column">
                <?php foreach ($rightServices as $id): ?>
                    <form method="post" class="clinic-card">
                        <input type="hidden" name="action" value="choose_service">
                        <input type="hidden" name="service_id" value="<?= (int)$id ?>">
                        <button type="submit" class="clinic-btn">
                            <span><?= htmlspecialchars($serviceNameMap[$id]) ?></span>
                            <span class="small">الحرف: <?= htmlspecialchars($serviceLetterMap[$id]) ?></span>
                        </button>
                    </form>
                <?php endforeach; ?>
            </div>

        </div>

    <?php endif; ?>
</div>
</body>
</html>
